import os
import time
import copy
import argparse
import numpy as np
import torch
import wandb
# import logging
import torch.nn as nn
from torchvision.utils import save_image
import torchvision
from utils import get_loops, get_dataset, get_network, get_eval_pool, evaluate_synset, evaluate_synset_old, get_daparam, match_loss, get_time, TensorDataset, epoch, DiffAugment, ParamDiffAug

from datetime import datetime

import torch.nn.functional as F

from resnet_big import SupConResNet

from PIL import PngImagePlugin
LARGE_ENOUGH_NUMBER = 100
PngImagePlugin.MAX_TEXT_CHUNK = LARGE_ENOUGH_NUMBER * (1024**2)

def main():

    parser = argparse.ArgumentParser(description='Parameter Processing')
    parser.add_argument('--method', type=str, default='DSA', help='DC/DSA')
    parser.add_argument('--dataset', type=str, default='CIFAR10', help='dataset')
    parser.add_argument('--conflict_percent', type=str, default='0.5', help='conflict sample percentage')
    parser.add_argument('--model', type=str, default='ConvNet', help='model')
    parser.add_argument('--ipc', type=int, default=1, help='image(s) per class')
    parser.add_argument('--eval_mode', type=str, default='S', help='eval_mode') # S: the same to training model, M: multi architectures,  W: net width, D: net depth, A: activation function, P: pooling layer, N: normalization layer,
    parser.add_argument('--num_exp', type=int, default=1, help='the number of experiments')
    parser.add_argument('--num_eval', type=int, default=1, help='the number of evaluating randomly initialized models')
    parser.add_argument('--epoch_eval_train', type=int, default=300, help='epochs to train a model with synthetic data')
    parser.add_argument('--Iteration', type=int, default=3000, help='training iterations')
    parser.add_argument('--eval_interval', type=int, default=100, help='how often to evaluate')
    parser.add_argument('--lr_img', type=float, default=0.1, help='learning rate for updating synthetic images')
    parser.add_argument('--lr_net', type=float, default=0.01, help='learning rate for updating network parameters')
    parser.add_argument('--batch_real', type=int, default=256, help='batch size for real data')
    parser.add_argument('--batch_train', type=int, default=256, help='batch size for training networks')
    parser.add_argument('--init', type=str, default='noise', help='noise/real: initialize synthetic images from random noise or randomly sampled real images.')
    parser.add_argument('--dsa_strategy', type=str, default='None', help='differentiable Siamese augmentation strategy')
    parser.add_argument('--data_path', type=str, default='data', help='dataset path')
    parser.add_argument('--save_path', type=str, default='result', help='path to save results')
    parser.add_argument('--dis_metric', type=str, default='ours', help='distance metric')
    parser.add_argument('--consine_similarity', action='store_true', help='whether to apply consine similarity')
    parser.add_argument('--consine_similarity_weight', type=float, default=0.2, help='weights for consine similarity')
    parser.add_argument('--bias_model', type=str, default=None, help='use bias model for similarity check')
    parser.add_argument('--log_table', type=int, default=1, help="how often to log the table.")
    parser.add_argument('--sim_threshold', type=float, default=0.2, help='learning rate for updating network parameters')
    parser.add_argument('--supcon_model_path', type=str, default='', help='path to save results')
    parser.add_argument('--use_kde', action='store_true', help='whether to use kde to compute the sample weights.')
    parser.add_argument('--cutoff_score', type=float, default=130, help='cutting off score for smoothing')

    args = parser.parse_args()
    args.outer_loop, args.inner_loop = get_loops(args.ipc)

    args.device = 'cuda' if torch.cuda.is_available() else 'cpu'
    args.dsa_param = ParamDiffAug()
    args.dsa = True if args.method == 'DSA' else False
    # logging.basicConfig(filename= args.method + args.dataset + '_IPC' + str(args.ipc) + '.log', filemode='w', level=logging.INFO, 
    #     format='%(message)s')

    if not os.path.exists(args.data_path):
        os.mkdir(args.data_path)

    if not os.path.exists(args.save_path):
        os.mkdir(args.save_path)

    eval_it_pool = np.arange(0, args.Iteration+1, args.eval_interval).tolist() if args.eval_mode == 'S' or args.eval_mode == 'SS' else [args.Iteration] # The list of iterations when we evaluate models and record results.
    channel, im_size, num_classes, class_names, mean, std, dst_train, dst_test, testloader = get_dataset(args.dataset, args.data_path, args.conflict_percent)
    model_eval_pool = get_eval_pool(args.eval_mode, args.model, args.model)

    accs_all_exps = dict() # record performances of all experiments
    for key in model_eval_pool:
        accs_all_exps[key] = []

    data_save = []

    run_name = datetime.now().strftime("%m/%d/%Y, %H:%M:%S") + '_' + args.method + '_' + args.dataset + '_' + args.model + '_pct' + str(args.conflict_percent) + '_ipc_' + str(args.ipc)
    if args.consine_similarity:
        run_name = run_name + '_biased_model_sim_' + str(args.consine_similarity_weight)
    wandb.init(sync_tensorboard=False, project='DebiasingDC', job_type='CleanRepo', config=args, name=run_name)

    # args = type('', (), {})()

    # for key in wandb.config._items:
    #     setattr(args, key, wandb.config._items[key])
    # args.dsa_param = ParamDiffAug()
    if args.bias_model:
        biased_model = get_network(args.model, channel, num_classes, im_size).to(args.device) # get a random model
        # for param in list(biased_model.parameters()):
        #     param.requires_grad = False
        biased_model.load_state_dict(torch.load('model/' + args.bias_model))
        biased_model.train()
        for param in list(biased_model.parameters()):
            param.requires_grad = False


    def compute_kde(sim_feature, true_bias_label, temp=0.1):
        kernel = torch.cdist(sim_feature, sim_feature, compute_mode='donot_use_mm_for_euclid_dist')
        kernel = torch.exp(-(kernel**2)/0.1)
        kernel = (1/kernel.mean(dim=-1))

        # normalize the weights to 1
        # breakpoint()
        reblanced_weight = 1/(1+torch.exp(kernel-args.cutoff_score))
        reblanced_weight = torch.nn.functional.softmax(reblanced_weight/temp)
        return reblanced_weight

    for exp in range(args.num_exp):
        print('\n================== Exp %d ==================\n '%exp)
        print('Hyper-parameters: \n', args.__dict__)
        print('Evaluation model pool: ', model_eval_pool)

        ''' organize the real dataset '''
        images_all = []
        labels_all = []
        indices_class = [[] for c in range(num_classes)]

        images_all = [torch.unsqueeze(dst_train[i][0], dim=0) for i in range(len(dst_train))]
        labels_all = [dst_train[i][1] for i in range(len(dst_train))]
        bias_labels_all = [dst_train[i][2] for i in range(len(dst_train))]
        for i, lab in enumerate(labels_all):
            indices_class[lab].append(i)
        images_all = torch.cat(images_all, dim=0).to(args.device)
        labels_all = torch.tensor(labels_all, dtype=torch.long, device=args.device)
        bias_labels_all = torch.tensor(bias_labels_all, dtype=torch.long, device=args.device)

        for c in range(num_classes):
            print('class c = %d: %d real images'%(c, len(indices_class[c])))

        def get_images(c, n): # get random n images from class c
            idx_shuffle = np.random.permutation(indices_class[c])[:n]
            return images_all[idx_shuffle]
        
        def get_images_and_labels(c, n): # get random n images from class c and their labels.
            idx_shuffle = np.random.permutation(indices_class[c])[:n]
            return images_all[idx_shuffle], labels_all[idx_shuffle], bias_labels_all[idx_shuffle]

        for ch in range(channel):
            print('real images channel %d, mean = %.4f, std = %.4f'%(ch, torch.mean(images_all[:, ch]), torch.std(images_all[:, ch])))


        ''' initialize the synthetic data '''
        image_syn = torch.randn(size=(num_classes*args.ipc, channel, im_size[0], im_size[1]), dtype=torch.float, requires_grad=True, device=args.device)
        label_syn = torch.tensor([np.ones(args.ipc)*i for i in range(num_classes)], dtype=torch.long, requires_grad=False, device=args.device).view(-1) # [0,0,0, 1,1,1, ..., 9,9,9]

        if args.init == 'real':
            print('initialize synthetic data from random real images')
            for c in range(num_classes):
                image_syn.data[c*args.ipc:(c+1)*args.ipc] = get_images(c, args.ipc).detach().data
        else:
            print('initialize synthetic data from random noise')


        ''' training '''
        optimizer_img = torch.optim.SGD([image_syn, ], lr=args.lr_img, momentum=0.5) # optimizer_img for synthetic data
        optimizer_img.zero_grad()
        criterion = nn.CrossEntropyLoss().to(args.device)
        criterion_reweight = nn.CrossEntropyLoss(reduction='none').to(args.device)
        print('%s training begins'%get_time())

        max_accuracy = 0.0

        if args.use_sim_matrix or args.use_kde:
            model_c = SupConResNet(name='resnet18').to(args.device)
            model_c.load_state_dict(torch.load(args.supcon_model_path, map_location='cuda:0')['state_dict'])
        

        for it in range(args.Iteration+1):

            wandb.log({"Progress" : it}, step=it)

            ''' Evaluate synthetic data '''
            if it in eval_it_pool:
                for model_eval in model_eval_pool:
                    print('-------------------------\nEvaluation\nmodel_train = %s, model_eval = %s, iteration = %d'%(args.model, model_eval, it))
                    if args.dsa:
                        args.epoch_eval_train = 1000
                        args.dc_aug_param = None
                        print('DSA augmentation strategy: \n', args.dsa_strategy)
                        print('DSA augmentation parameters: \n', args.dsa_param.__dict__)
                    else:
                        args.dc_aug_param = get_daparam(args.dataset, model_eval) # This augmentation parameter set is only for DC method. It will be muted when args.dsa is True.
                        print('DC augmentation parameters: \n', args.dc_aug_param)

                    if args.dsa or args.dc_aug_param['strategy'] != 'none':
                        args.epoch_eval_train = 1000  # Training with data augmentation needs more epochs.
                    else:
                        args.epoch_eval_train = 300

                    accs = []
                    use_preselected = True
                    for it_eval in range(args.num_eval):
                        net_eval = get_network(model_eval, channel, num_classes, im_size).to(args.device) # get a random model
                        image_syn_eval, label_syn_eval = copy.deepcopy(image_syn.detach()), copy.deepcopy(label_syn.detach()) # avoid any unaware modification
                        _, acc_train, acc_test = evaluate_synset_old(it_eval, net_eval, image_syn_eval, label_syn_eval, testloader, args)
                        accs.append(acc_test)
                    print('Evaluate %d random %s, mean = %.4f std = %.4f\n-------------------------'%(len(accs), model_eval, np.mean(accs), np.std(accs)))
                    # logging.info('%d\t%.4f\t%.4f'%(it, np.mean(accs), np.std(accs)))
                    current_accuracy = np.mean(accs)
                    wandb.log({"Accuracy/{}".format(model_eval) : current_accuracy}, step=it)
                    wandb.log({"Std/{}".format(model_eval) : np.std(accs)}, step=it)
                    max_accuracy = max(max_accuracy, current_accuracy)
                    wandb.log({"Max Accuracy/{}".format(model_eval) : max_accuracy}, step=it)

                    if it == args.Iteration: # record the final results
                        accs_all_exps[model_eval] += accs

                ''' visualize and save '''
                save_name = os.path.join(args.save_path, 'vis_%s_%s_pct%s_%s_%dipc_exp%d_iter%d.png'%(args.method, args.dataset, args.conflict_percent, args.model, args.ipc, exp, it))
                image_syn_vis = copy.deepcopy(image_syn.detach().cpu())
                for ch in range(channel):
                    image_syn_vis[:, ch] = image_syn_vis[:, ch]  * std[ch] + mean[ch]
                image_syn_vis[image_syn_vis<0] = 0.0
                image_syn_vis[image_syn_vis>1] = 1.0
                save_image(image_syn_vis, save_name, nrow=args.ipc) # Trying normalize = True/False may get better visual effects.
                if args.ipc < 50:
                    upsampled = torch.repeat_interleave(image_syn_vis, repeats=4, dim=2)
                    upsampled = torch.repeat_interleave(upsampled, repeats=4, dim=3)
                    grid = torchvision.utils.make_grid(upsampled, nrow=10, normalize=True, scale_each=True)
                    wandb.log({"Reconstructed_Images": wandb.Image(torch.nan_to_num(grid.detach().cpu()))}, step=it)



            ''' Train synthetic data '''
            net = get_network(args.model, channel, num_classes, im_size).to(args.device) # get a random model
            net.train()
            net_parameters = list(net.parameters())
            optimizer_net = torch.optim.SGD(net.parameters(), lr=args.lr_net)  # optimizer_img for synthetic data
            optimizer_net.zero_grad()
            loss_avg = 0
            args.dc_aug_param = None  # Mute the DC augmentation when learning synthetic data (in inner-loop epoch function) in oder to be consistent with DC paper.

            for ol in range(args.outer_loop):

                ''' freeze the running mu and sigma for BatchNorm layers '''
                # Synthetic data batch, e.g. only 1 image/batch, is too small to obtain stable mu and sigma.
                # So, we calculate and freeze mu and sigma for BatchNorm layer with real data batch ahead.
                # This would make the training with BatchNorm layers easier.

                BN_flag = False
                BNSizePC = 16  # for batch normalization
                for module in net.modules():
                    if 'BatchNorm' in module._get_name(): #BatchNorm
                        BN_flag = True
                if BN_flag:
                    img_real = torch.cat([get_images(c, BNSizePC) for c in range(num_classes)], dim=0)
                    net.train() # for updating the mu, sigma of BatchNorm
                    output_real = net(img_real) # get running mu, sigma
                    for module in net.modules():
                        if 'BatchNorm' in module._get_name():  #BatchNorm
                            module.eval() # fix mu and sigma of every BatchNorm layer


                similarity = 0.0
                ''' update synthetic data '''
                for c in range(num_classes):
                    loss = torch.tensor(0.0).to(args.device)
                    img_real, labels_real, bias_labels_real = get_images_and_labels(c, args.batch_real)
                    lab_real = torch.ones((img_real.shape[0],), device=args.device, dtype=torch.long) * c
                    img_syn = image_syn[c*args.ipc:(c+1)*args.ipc].reshape((args.ipc, channel, im_size[0], im_size[1]))
                    lab_syn = torch.ones((args.ipc,), device=args.device, dtype=torch.long) * c

                    if args.dsa:
                        seed = int(time.time() * 1000) % 100000
                        img_real = DiffAugment(img_real, args.dsa_strategy, seed=seed, param=args.dsa_param)
                        img_syn = DiffAugment(img_syn, args.dsa_strategy, seed=seed, param=args.dsa_param)

                    output_real = net(img_real)
                    if args.use_kde:
                        with torch.no_grad():
                            sim_feature = model_c(img_real)
                            reblanced_weight = compute_kde(sim_feature, bias_labels_real, 0.1)
                        loss_real = criterion_reweight(output_real, lab_real)
                        loss_real = reblanced_weight.dot(loss_real)
                    else:
                        loss_real = criterion(output_real, lab_real)
                    gw_real = torch.autograd.grad(loss_real, net_parameters)
                    gw_real = list((_.detach().clone() for _ in gw_real))

                    output_syn = net(img_syn)
                    loss_syn = criterion(output_syn, lab_syn)
                    gw_syn = torch.autograd.grad(loss_syn, net_parameters, create_graph=True)

                    loss = match_loss(gw_syn, gw_real, args)

                    optimizer_img.zero_grad()
                    loss.backward()
                    optimizer_img.step()
                    loss_avg += loss.item()


                if ol == args.outer_loop - 1:
                    break

                ''' update network '''
                image_syn_train, label_syn_train = copy.deepcopy(image_syn.detach()), copy.deepcopy(label_syn.detach())  # avoid any unaware modification
                dst_syn_train = TensorDataset(image_syn_train, label_syn_train)
                trainloader = torch.utils.data.DataLoader(dst_syn_train, batch_size=args.batch_train, shuffle=True, num_workers=0)
                for il in range(args.inner_loop):
                    epoch('train', trainloader, net, optimizer_net, criterion, args, aug = True if args.dsa else False)

            if args.bias_model:
                wandb.log({'Similarity:': similarity / num_classes}, step=it)
            loss_avg /= (num_classes*args.outer_loop)

            if it%10 == 0:
                print('%s iter = %04d, loss = %.4f' % (get_time(), it, loss_avg))
                wandb.log({'Matching loss:': loss_avg}, step=it)

            if it == args.Iteration: # only record the final results
                data_save.append([copy.deepcopy(image_syn.detach().cpu()), copy.deepcopy(label_syn.detach().cpu())])
                torch.save({'data': data_save, 'accs_all_exps': accs_all_exps, }, os.path.join(args.save_path, 'res_%s_%s_%s_%dipc.pt'%(args.method, args.dataset, args.model, args.ipc)))


    print('\n==================== Final Results ====================\n')
    for key in model_eval_pool:
        accs = accs_all_exps[key]
        print('Run %d experiments, train on %s, evaluate %d random %s, mean  = %.2f%%  std = %.2f%%'%(args.num_exp, args.model, len(accs), key, np.mean(accs)*100, np.std(accs)*100))
    
    wandb.finish()



if __name__ == '__main__':
    main()


